"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const ts = require("typescript");
function transformJavascript(options) {
    const { content, getTransforms, emitSourceMap, inputFilePath, outputFilePath, strict, } = options;
    // Bail if there's no transform to do.
    if (getTransforms.length === 0) {
        return {
            content: null,
            sourceMap: null,
            emitSkipped: true,
        };
    }
    // Print error diagnostics.
    const checkDiagnostics = (diagnostics) => {
        if (diagnostics && diagnostics.length > 0) {
            let errors = '';
            errors = errors + '\n' + ts.formatDiagnostics(diagnostics, {
                getCurrentDirectory: () => ts.sys.getCurrentDirectory(),
                getNewLine: () => ts.sys.newLine,
                getCanonicalFileName: (f) => f,
            });
            return errors;
        }
    };
    const outputs = new Map();
    const tempFilename = 'bo-default-file.js';
    const tempSourceFile = ts.createSourceFile(tempFilename, content, ts.ScriptTarget.Latest);
    const host = {
        getSourceFile: (fileName) => {
            if (fileName !== tempFilename) {
                throw new Error(`File ${fileName} does not have a sourceFile.`);
            }
            return tempSourceFile;
        },
        getDefaultLibFileName: () => 'lib.d.ts',
        getCurrentDirectory: () => '',
        getDirectories: () => [],
        getCanonicalFileName: (fileName) => fileName,
        useCaseSensitiveFileNames: () => true,
        getNewLine: () => '\n',
        fileExists: (fileName) => fileName === tempFilename,
        readFile: (_fileName) => '',
        writeFile: (fileName, text) => outputs.set(fileName, text),
    };
    const tsOptions = {
        // We target latest so that there is no downleveling.
        target: ts.ScriptTarget.Latest,
        isolatedModules: true,
        suppressOutputPathCheck: true,
        allowNonTsExtensions: true,
        noLib: true,
        noResolve: true,
        sourceMap: emitSourceMap,
        inlineSources: emitSourceMap,
        inlineSourceMap: false,
    };
    const program = ts.createProgram([tempFilename], tsOptions, host);
    const diagnostics = program.getSyntacticDiagnostics(tempSourceFile);
    const hasError = diagnostics.some(diag => diag.category === ts.DiagnosticCategory.Error);
    if (hasError) {
        // Throw only if we're in strict mode, otherwise return original content.
        if (strict) {
            throw new Error(`
        TS failed with the following error messages:

        ${checkDiagnostics(diagnostics)}
      `);
        }
        else {
            return {
                content: null,
                sourceMap: null,
                emitSkipped: true,
            };
        }
    }
    // We need the checker inside transforms.
    const transforms = getTransforms.map((getTf) => getTf(program));
    program.emit(undefined, undefined, undefined, undefined, { before: transforms, after: [] });
    let transformedContent = outputs.get(tempFilename);
    if (!transformedContent) {
        return {
            content: null,
            sourceMap: null,
            emitSkipped: true,
        };
    }
    let sourceMap = null;
    const tsSourceMap = outputs.get(`${tempFilename}.map`);
    if (emitSourceMap && tsSourceMap) {
        const urlRegExp = /^\/\/# sourceMappingURL=[^\r\n]*/gm;
        sourceMap = JSON.parse(tsSourceMap);
        // Fix sourcemaps file references.
        if (outputFilePath) {
            sourceMap.file = outputFilePath;
            transformedContent = transformedContent.replace(urlRegExp, `//# sourceMappingURL=${sourceMap.file}.map\n`);
            if (inputFilePath) {
                sourceMap.sources = [inputFilePath];
            }
            else {
                sourceMap.sources = [''];
            }
        }
        else {
            // TODO: figure out if we should inline sources here.
            transformedContent = transformedContent.replace(urlRegExp, '');
            sourceMap.file = '';
            sourceMap.sources = [''];
        }
    }
    return {
        content: transformedContent,
        sourceMap,
        emitSkipped: false,
    };
}
exports.transformJavascript = transformJavascript;
//# sourceMappingURL=data:application/json;base64,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