"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var tslib_1 = require("tslib");
var ts = require("typescript");
var Lint = require("tslint");
var OPTION_ALWAYS = 'always';
var RULE_NAME = 'array-bracket-spacing';
var Rule = (function (_super) {
    tslib_1.__extends(Rule, _super);
    function Rule() {
        return _super !== null && _super.apply(this, arguments) || this;
    }
    Rule.prototype.apply = function (sourceFile) {
        var walker = new ArrayBracketSpacingWalker(sourceFile, this.getOptions());
        return this.applyWithWalker(walker);
    };
    return Rule;
}(Lint.Rules.AbstractRule));
Rule.metadata = {
    ruleName: RULE_NAME,
    description: 'enforce consistent spacing inside array brackets',
    rationale: (_a = ["\n      A number of style guides require or disallow spaces between array brackets and other tokens.\n      This rule applies to both array literals and destructuring assignments (ECMAScript 6).\n      "], _a.raw = ["\n      A number of style guides require or disallow spaces between array brackets and other tokens.\n      This rule applies to both array literals and destructuring assignments (ECMAScript 6).\n      "], Lint.Utils.dedent(_a)),
    optionsDescription: (_b = ["\n      The rule takes one or two options. The first is a string, which can be:\n\n      - `\"never\"` (default) disallows spaces inside array brackets\n      - `\"always\"`requires one or more spaces or newlines inside array brackets\n\n      The second option is an object for exceptions to the `\"never\"` option:\n\n      - `\"singleValue\": true` requires one or more spaces or newlines inside brackets of array\n                                literals that contain a single element\n      - `\"objectsInArrays\": true` requires one or more spaces or newlines between brackets of\n                                    array literals and braces of their object literal elements\n                                    `[ {` or `} ]`\n      - `\"arraysInArrays\": true` requires one or more spaces or newlines between brackets of\n                                   array literals and brackets of their array literal elements\n                                   `[ [` or `] ]`\n\n      When using the `\"always\"` option the second option takes on these exceptions:\n\n      - `\"singleValue\": false` disallows spaces inside brackets of array literals that contain a\n                                 single element\n      - `\"objectsInArrays\": false` disallows spaces between brackets of array literals and braces\n                                     of their object literal elements `[ {` or `} ]`\n      - `\"arraysInArrays\": false` disallows spaces between brackets of array literals and brackets\n                                    of their array literal elements `[ [` or `] ]`\n\n      This rule has build-in exceptions:\n\n      - `\"never\"` (and also the exceptions to the `\"always\"` option) allows newlines inside\n                    array brackets, because this is a common pattern\n      - `\"always\"` does not require spaces or newlines in empty array literals `[]`\n      "], _b.raw = ["\n      The rule takes one or two options. The first is a string, which can be:\n\n      - \\`\"never\"\\` (default) disallows spaces inside array brackets\n      - \\`\"always\"\\`requires one or more spaces or newlines inside array brackets\n\n      The second option is an object for exceptions to the \\`\"never\"\\` option:\n\n      - \\`\"singleValue\": true\\` requires one or more spaces or newlines inside brackets of array\n                                literals that contain a single element\n      - \\`\"objectsInArrays\": true\\` requires one or more spaces or newlines between brackets of\n                                    array literals and braces of their object literal elements\n                                    \\`[ {\\` or \\`} ]\\`\n      - \\`\"arraysInArrays\": true\\` requires one or more spaces or newlines between brackets of\n                                   array literals and brackets of their array literal elements\n                                   \\`[ [\\` or \\`] ]\\`\n\n      When using the \\`\"always\"\\` option the second option takes on these exceptions:\n\n      - \\`\"singleValue\": false\\` disallows spaces inside brackets of array literals that contain a\n                                 single element\n      - \\`\"objectsInArrays\": false\\` disallows spaces between brackets of array literals and braces\n                                     of their object literal elements \\`[ {\\` or \\`} ]\\`\n      - \\`\"arraysInArrays\": false\\` disallows spaces between brackets of array literals and brackets\n                                    of their array literal elements \\`[ [\\` or \\`] ]\\`\n\n      This rule has build-in exceptions:\n\n      - \\`\"never\"\\` (and also the exceptions to the \\`\"always\"\\` option) allows newlines inside\n                    array brackets, because this is a common pattern\n      - \\`\"always\"\\` does not require spaces or newlines in empty array literals \\`[]\\`\n      "], Lint.Utils.dedent(_b)),
    options: {
        anyOf: [
            {
                type: 'array',
                items: [
                    {
                        enum: ['always', 'never']
                    }
                ],
                minItems: 0,
                maxItems: 1
            },
            {
                type: 'object',
                properties: {
                    singleValue: {
                        type: 'boolean'
                    },
                    objectsInArrays: {
                        type: 'boolean'
                    },
                    arraysInArrays: {
                        type: 'boolean'
                    }
                },
                additionalProperties: false
            }
        ]
    },
    optionExamples: [
        (_c = ["\n        \"", "\": [true, \"always\"]\n        "], _c.raw = ["\n        \"", "\": [true, \"always\"]\n        "], Lint.Utils.dedent(_c, RULE_NAME)),
        (_d = ["\n        \"", "\": [true, \"never\"]\n        "], _d.raw = ["\n        \"", "\": [true, \"never\"]\n        "], Lint.Utils.dedent(_d, RULE_NAME)),
        (_e = ["\n        \"", "\": [true, \"never\", {\n          \"arraysInArrays\": true\n        }]\n        "], _e.raw = ["\n        \"", "\": [true, \"never\", {\n          \"arraysInArrays\": true\n        }]\n        "], Lint.Utils.dedent(_e, RULE_NAME))
    ],
    typescriptOnly: false,
    type: 'style'
};
exports.Rule = Rule;
var ArrayBracketSpacingWalker = (function (_super) {
    tslib_1.__extends(ArrayBracketSpacingWalker, _super);
    function ArrayBracketSpacingWalker(sourceFile, options) {
        var _this = _super.call(this, sourceFile, options) || this;
        _this.singleValueException = false;
        _this.objectsInArraysException = false;
        _this.arraysInArraysException = false;
        var ruleOptions = _this.getOptions();
        _this.spaced = _this.hasOption(OPTION_ALWAYS) || (ruleOptions && ruleOptions.length === 0);
        var opt = ruleOptions[1];
        var isDef = function (x) { return typeof x !== 'undefined'; };
        if (opt) {
            _this.singleValueException = isDef(opt.singleValue) && opt.singleValue !== _this.spaced;
            _this.objectsInArraysException = isDef(opt.objectsInArrays) && opt.objectsInArrays !== _this.spaced;
            _this.arraysInArraysException = isDef(opt.arraysInArrays) && opt.arraysInArrays !== _this.spaced;
        }
        return _this;
    }
    ArrayBracketSpacingWalker.prototype.report = function (start, msg, fix) {
        this.addFailure(this.createFailure(start, 1, msg, fix));
    };
    ArrayBracketSpacingWalker.prototype.reportNoBeginningSpace = function (token, space) {
        var start = token.getStart(this.getSourceFile());
        var fix = Lint.Replacement.deleteText(start + 1, space);
        this.report(start, 'There should be no space after "["', fix);
    };
    ArrayBracketSpacingWalker.prototype.reportRequiredBeginningSpace = function (token) {
        var start = token.getStart(this.getSourceFile());
        var fix = Lint.Replacement.appendText(start + 1, ' ');
        this.report(start, 'A space is required after "["', fix);
    };
    ArrayBracketSpacingWalker.prototype.reportRequiredEndingSpace = function (token) {
        var start = token.getStart(this.getSourceFile());
        var fix = Lint.Replacement.appendText(start, ' ');
        this.report(start, 'A space is required before "]"', fix);
    };
    ArrayBracketSpacingWalker.prototype.reportNoEndingSpace = function (token, space) {
        var start = token.getStart(this.getSourceFile());
        var fix = Lint.Replacement.deleteText(start - space, space);
        this.report(start, 'There should be no space before "]"', fix);
    };
    ArrayBracketSpacingWalker.prototype.visitNode = function (node) {
        if (node.kind === ts.SyntaxKind.ArrayBindingPattern) {
            this.validateArraySpacing(node, node.elements);
        }
        _super.prototype.visitNode.call(this, node);
    };
    ArrayBracketSpacingWalker.prototype.visitArrayLiteralExpression = function (node) {
        this.validateArraySpacing(node, node.elements);
        _super.prototype.visitArrayLiteralExpression.call(this, node);
    };
    ArrayBracketSpacingWalker.prototype.isObjectType = function (node) {
        return node && node.kind === ts.SyntaxKind.ObjectLiteralExpression;
    };
    ArrayBracketSpacingWalker.prototype.isArrayType = function (node) {
        if (node) {
            if (node.kind === ts.SyntaxKind.ArrayLiteralExpression) {
                return true;
            }
            var firstChild = node.getChildAt(0);
            if (firstChild && firstChild.kind === ts.SyntaxKind.ArrayBindingPattern) {
                return true;
            }
        }
        return false;
    };
    ArrayBracketSpacingWalker.prototype.validateArraySpacing = function (node, elements) {
        var _this = this;
        if (this.spaced && elements.length === 0) {
            return;
        }
        var first = node.getChildAt(0);
        var last = node.getChildAt(2);
        var firstElement = elements[0];
        var lastElement = elements[elements.length - 1];
        var second = firstElement || last;
        var penultimate = lastElement || first;
        if (second.pos === second.end) {
            second = node.getChildAt(1).getChildAt(1);
        }
        if (elements.hasTrailingComma) {
            penultimate = elements;
        }
        var mustBeSpaced = function (token) { return (_this.singleValueException && elements.length === 1 ||
            _this.objectsInArraysException && _this.isObjectType(token) ||
            _this.arraysInArraysException && _this.isArrayType(token)) ? !_this.spaced : _this.spaced; };
        var openingBracketMustBeSpaced = mustBeSpaced(firstElement);
        var closingBracketMustBeSpaced = mustBeSpaced(lastElement);
        var spaceAfterOpeningBracket = this.getSpaceBetween(first, second, false);
        var isBreakAfterOpeningBracket = this.isLineBreakBetween(first, second);
        var spaceBeforeClosingBracket = this.getSpaceBetween(penultimate, last, true);
        var isBreakBeforeClosingBracket = this.isLineBreakBetween(penultimate, last);
        if (!isBreakAfterOpeningBracket) {
            if (openingBracketMustBeSpaced && !spaceAfterOpeningBracket) {
                this.reportRequiredBeginningSpace(first);
            }
            else if (!openingBracketMustBeSpaced && spaceAfterOpeningBracket) {
                this.reportNoBeginningSpace(first, spaceAfterOpeningBracket);
            }
        }
        if (first !== penultimate && !isBreakBeforeClosingBracket) {
            if (closingBracketMustBeSpaced && !spaceBeforeClosingBracket) {
                this.reportRequiredEndingSpace(last);
            }
            else if (!closingBracketMustBeSpaced && spaceBeforeClosingBracket) {
                this.reportNoEndingSpace(last, spaceBeforeClosingBracket);
            }
        }
    };
    ArrayBracketSpacingWalker.prototype.getSpaceBetween = function (node, nextNode, trailing) {
        var end = nextNode.getStart(this.getSourceFile());
        var start = node.end;
        var text = this.getSourceFile().text.substring(start, end);
        var m = text.match(/\/\*.*\*\//);
        if (m && typeof m.index === 'number') {
            var len = m[0].length;
            return trailing ? end - (start + m.index + len) : m.index;
        }
        return end - start;
    };
    ArrayBracketSpacingWalker.prototype.isLineBreakBetween = function (node, nextNode) {
        return this.getEndPosition(node).line !== this.getStartPosition(nextNode).line;
    };
    ArrayBracketSpacingWalker.prototype.getStartPosition = function (node) {
        var srcFile = this.getSourceFile();
        return srcFile.getLineAndCharacterOfPosition(node.getStart(srcFile));
    };
    ArrayBracketSpacingWalker.prototype.getEndPosition = function (node) {
        return this.getSourceFile().getLineAndCharacterOfPosition(node.end);
    };
    return ArrayBracketSpacingWalker;
}(Lint.RuleWalker));
var _a, _b, _c, _d, _e;

//# sourceMappingURL=data:application/json;charset=utf8;base64,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
