"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
var tslib_1 = require("tslib");
var Lint = require("tslint");
var ts = require("typescript");
var RULE_NAME = 'sort-imports';
var Rule = (function (_super) {
    tslib_1.__extends(Rule, _super);
    function Rule() {
        return _super !== null && _super.apply(this, arguments) || this;
    }
    Rule.prototype.apply = function (sourceFile) {
        var walker = new RuleWalker(sourceFile, this.getOptions());
        return this.applyWithWalker(walker);
    };
    return Rule;
}(Lint.Rules.AbstractRule));
Rule.metadata = {
    ruleName: RULE_NAME,
    description: 'enforce sorting import declarations within module',
    rationale: (_a = ["\n      When declaring multiple imports, a sorted list of import declarations make it easier for developers to\n      read the code and find necessary imports later. This rule is purely a matter of style.\n\n      This rule checks all import declarations and verifies that all imports are first sorted by the used member\n      syntax and then alphabetically by the first member or alias name.\n      "], _a.raw = ["\n      When declaring multiple imports, a sorted list of import declarations make it easier for developers to\n      read the code and find necessary imports later. This rule is purely a matter of style.\n\n      This rule checks all import declarations and verifies that all imports are first sorted by the used member\n      syntax and then alphabetically by the first member or alias name.\n      "], Lint.Utils.dedent(_a)),
    optionsDescription: (_b = ["\n      - `\"ignore-case\"` does case-insensitive comparisons (default: `false`)\n      - `\"ignore-member-sort\"` allows members in multiple type imports to occur in any order (default: `false`)\n      - `\"member-syntax-sort-order\"` (default: `[\"none\", \"all\", \"multiple\", \"single\", \"alias\"]`); all 5 items must be\n      present in the array, but you can change the order:\n        - `none` = import module without exported bindings.\n        - `all` = import all members provided by exported bindings.\n        - `multiple` = import multiple members.\n        - `single` = import a single member.\n        - `alias` = creates an alias for a member. This is unique to TER and not in ESLint's `sort-imports`.\n      "], _b.raw = ["\n      - \\`\"ignore-case\"\\` does case-insensitive comparisons (default: \\`false\\`)\n      - \\`\"ignore-member-sort\"\\` allows members in multiple type imports to occur in any order (default: \\`false\\`)\n      - \\`\"member-syntax-sort-order\"\\` (default: \\`[\"none\", \"all\", \"multiple\", \"single\", \"alias\"]\\`); all 5 items must be\n      present in the array, but you can change the order:\n        - \\`none\\` = import module without exported bindings.\n        - \\`all\\` = import all members provided by exported bindings.\n        - \\`multiple\\` = import multiple members.\n        - \\`single\\` = import a single member.\n        - \\`alias\\` = creates an alias for a member. This is unique to TER and not in ESLint's \\`sort-imports\\`.\n      "], Lint.Utils.dedent(_b)),
    options: {
        type: 'object',
        properties: {
            'member-syntax-sort-order': {
                type: 'array',
                items: {
                    type: 'string',
                    enum: ['none', 'all', 'multiple', 'single', 'alias']
                },
                minLength: 5,
                maxLength: 5
            },
            'ignore-case': {
                type: 'boolean'
            },
            'ignore-member-sort': {
                type: 'boolean'
            }
        }
    },
    optionExamples: [
        (_c = ["\n        \"", "\": [true]\n        "], _c.raw = ["\n        \"", "\": [true]\n        "], Lint.Utils.dedent(_c, RULE_NAME)),
        (_d = ["\n        \"", "\": [true, { \"ignore-case\" }]\n        "], _d.raw = ["\n        \"", "\": [true, { \"ignore-case\" }]\n        "], Lint.Utils.dedent(_d, RULE_NAME)),
        (_e = ["\n        \"", "\": [true, { \"ignore-member-sort\" }]\n        "], _e.raw = ["\n        \"", "\": [true, { \"ignore-member-sort\" }]\n        "], Lint.Utils.dedent(_e, RULE_NAME)),
        (_f = ["\n        \"", "\": [true, { \"member-syntax-sort-order\": [\"all\", \"single\", \"multiple\", \"none\", \"alias\"] }]\n        "], _f.raw = ["\n        \"", "\": [true, { \"member-syntax-sort-order\": [\"all\", \"single\", \"multiple\", \"none\", \"alias\"] }]\n        "], Lint.Utils.dedent(_f, RULE_NAME))
    ],
    typescriptOnly: false,
    type: 'style'
};
exports.Rule = Rule;
var MemberSyntaxType;
(function (MemberSyntaxType) {
    MemberSyntaxType[MemberSyntaxType["None"] = 0] = "None";
    MemberSyntaxType[MemberSyntaxType["All"] = 1] = "All";
    MemberSyntaxType[MemberSyntaxType["Multiple"] = 2] = "Multiple";
    MemberSyntaxType[MemberSyntaxType["Single"] = 3] = "Single";
    MemberSyntaxType[MemberSyntaxType["Alias"] = 4] = "Alias";
})(MemberSyntaxType || (MemberSyntaxType = {}));
var RuleWalker = (function (_super) {
    tslib_1.__extends(RuleWalker, _super);
    function RuleWalker(sourceFile, options) {
        var _this = _super.call(this, sourceFile, options) || this;
        _this.currentImportIndex = 0;
        var optionSet = _this.getOptions()[0] || {};
        _this.ignoreCase = _this.hasOption('ignore-case');
        _this.ignoreMemberSort = _this.hasOption('ignore-member-sort');
        _this.expectedOrder = RuleWalker._processMemberSyntaxSortOrder(optionSet['member-syntax-sort-order']);
        _this.currentSortValue = { sortValue: '', originalValue: '' };
        if (_this.ignoreCase) {
            _this.caseConverter = function (s) { return s.toUpperCase(); };
        }
        else {
            _this.caseConverter = function (s) { return s; };
        }
        return _this;
    }
    RuleWalker.prototype.visitNode = function (node) {
        if (node.kind === ts.SyntaxKind.ImportDeclaration ||
            node.kind === ts.SyntaxKind.ImportEqualsDeclaration) {
            this._validateOrder(node);
        }
        _super.prototype.visitNode.call(this, node);
    };
    RuleWalker.prototype.visitNamedImports = function (node) {
        if (!this.ignoreMemberSort) {
            this._validateMemberSort(node);
        }
        _super.prototype.visitNamedImports.call(this, node);
    };
    RuleWalker.prototype._validateMemberSort = function (node) {
        var _this = this;
        var imports = node.elements.map(function (e) { return _this.caseConverter(e.getText()); });
        var importReduction = imports.reduce(function (prev, current) { return prev + current; });
        var sortedImports = imports.sort();
        var sortedReduction = sortedImports.reduce(function (prev, current) { return prev + current; });
        if (importReduction !== sortedReduction) {
            this.addFailureAtNode(node, 'Member imports must be sorted alphabetically.');
        }
    };
    RuleWalker.prototype._validateOrder = function (node) {
        var importData = this._determineImportType(node);
        if (importData) {
            var index = this.expectedOrder.indexOf(importData.memberSyntaxType, this.currentImportIndex);
            if (index !== -1) {
                if (this.expectedOrder[this.currentImportIndex] !== importData.memberSyntaxType) {
                    this.currentImportIndex = index;
                    this.currentSortValue = {
                        sortValue: this.caseConverter(importData.sortValue),
                        originalValue: importData.sortValue
                    };
                }
                else if (this.currentSortValue.sortValue > this.caseConverter(importData.sortValue)) {
                    this.addFailureAtNode(node, "All imports of the same type must be sorted alphabetically. \"" + importData.sortValue + "\" must come before \"" + this.currentSortValue.originalValue + "\"");
                }
                else {
                    this.currentSortValue = {
                        sortValue: this.caseConverter(importData.sortValue),
                        originalValue: importData.sortValue
                    };
                }
            }
            else {
                var currentSyntaxType = MemberSyntaxType[importData.memberSyntaxType];
                var previousSyntaxType = MemberSyntaxType[this.expectedOrder[this.currentImportIndex]];
                this.addFailureAtNode(node, "All imports of type \"" + currentSyntaxType + "\" must occur before all imports of type \"" + previousSyntaxType + "\"");
            }
        }
        else {
            this.addFailureAtNode(node, 'Could not determine import type');
        }
    };
    RuleWalker.prototype._determineImportType = function (node) {
        var nodeText = node.getFullText();
        if (node.kind === ts.SyntaxKind.ImportEqualsDeclaration) {
            var aliasMatch = /\bimport\s+(\w+)\s*=.+/g.exec(nodeText);
            return {
                memberSyntaxType: MemberSyntaxType.Alias,
                sortValue: aliasMatch[1]
            };
        }
        else {
            var singleMatch = /\bimport\s+(?:{?([^,{}\*]+?)}?)\s*from\s+[\'"](?:[^"\']+)["\']/g.exec(nodeText);
            var multipleMatch = /\bimport\s*{?\s*([^{}\'",]+?)\s*,(?:\s*.+\s*,\s*)*\s*.+\s*}?\s*from\s+[\'"](?:[^"\']+)["\']/g.exec(nodeText);
            var noneMatch = /\bimport\s+[\'"]([^"\']+)["\']/g.exec(nodeText);
            var allMatch = /\bimport\s+\*\s+as\s+(.+)\s+from\s+[\'"](?:[^"\']+)["\']/g.exec(nodeText);
            var result = void 0;
            if (singleMatch !== null) {
                result = {
                    memberSyntaxType: MemberSyntaxType.Single,
                    sortValue: singleMatch[1]
                };
            }
            else if (multipleMatch !== null) {
                result = {
                    memberSyntaxType: MemberSyntaxType.Multiple,
                    sortValue: multipleMatch[1]
                };
            }
            else if (noneMatch !== null) {
                result = {
                    memberSyntaxType: MemberSyntaxType.None,
                    sortValue: noneMatch[1]
                };
            }
            else if (allMatch !== null) {
                result = {
                    memberSyntaxType: MemberSyntaxType.All,
                    sortValue: allMatch[1]
                };
            }
            else {
                result = {
                    memberSyntaxType: MemberSyntaxType.None,
                    sortValue: ''
                };
            }
            return result;
        }
    };
    RuleWalker._processMemberSyntaxSortOrder = function (sortOption) {
        var defaultOrder = [MemberSyntaxType.None, MemberSyntaxType.All, MemberSyntaxType.Multiple, MemberSyntaxType.Single, MemberSyntaxType.Alias];
        if (Array.isArray(sortOption) && typeof sortOption[0] === 'string' && sortOption.length === 5) {
            var memberSyntaxTypeMap_1 = {
                none: MemberSyntaxType.None,
                all: MemberSyntaxType.All,
                multiple: MemberSyntaxType.Multiple,
                single: MemberSyntaxType.Single,
                alias: MemberSyntaxType.Alias
            };
            var order_1 = [];
            var usedOptions_1 = {};
            sortOption.forEach(function (t) {
                if (usedOptions_1[t] !== undefined) {
                }
                else {
                    usedOptions_1[t] = t;
                    if (memberSyntaxTypeMap_1[t]) {
                        order_1.push(memberSyntaxTypeMap_1[t]);
                    }
                }
            });
            return order_1;
        }
        else {
            return defaultOrder;
        }
    };
    return RuleWalker;
}(Lint.RuleWalker));
var _a, _b, _c, _d, _e, _f;

//# sourceMappingURL=data:application/json;charset=utf8;base64,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
